/**
 *  FileSender provides methods that allow any file to be sent across
 *	a TCP/IP connection as a byte array.
 *
 *	@author Zhenlan Jin
 *	@version %I%, %G%
 *	@since 1.0
 */

import java.io.*;
import java.net.*;

public class FileSender {
	private String fileName;

	/**
	 *	Obtains the name of the file to be sent across to the client.
	 *	
	 *	@param	file	a string indicating the name of the file to be sent.
	 */
  	public FileSender(String file) {
  		fileName = file;
  	}
  	
  	/**
  	 *	Initializes the server.  This method will wait for a connection from
  	 *	a client and establish a TCP/IP connection.  Upon receiving a request
  	 *	from the client for a file, the byte array of the file will be sent by
  	 *	a call to the sendPic method.
  	 *	
  	 *	@param	void
  	 *	@return	void
  	 */
 	
 	public void runServer() {
 		InputStreamReader temp;
 		BufferedReader inFromClient;
 		DataOutputStream outToClient;
		int i = 0;

		try {
			ServerSocket startSocket = new ServerSocket(6789);
	
			while(true) {
				System.out.println("Waiting for clients");
				Socket connectionSocket = startSocket.accept();
	
				// create the inputstream and outputstream with the client.
				temp = new InputStreamReader(connectionSocket.getInputStream());
				inFromClient = new BufferedReader(temp);
				outToClient = new DataOutputStream(connectionSocket.getOutputStream());
			
				i = 0;
				while (!(connectionSocket.isClosed()) && (i != -1)) {		
					System.out.println("Waiting for client to send data");
				
					// read in the request from client
					i = inFromClient.read();
				
					// 200 means request from client to send file
					if (i == 200) {
						sendFile(outToClient, readFile(fileName));
					}
				}
			}
		}
		catch (IOException e) {
	  		System.out.println("Connection Error: connection with client terminated");
		}
 	}
 	
 	/**
 	 *	Reads in the data from the file specifited by String name into a byte
 	 *	array and returns it.
 	 *
 	 *	@param		name		the name of the file to be read
 	 *	@return					the buffer containing the file data.
 	 *	@exception	IOException
 	 */
    
    public byte[] readFile(String name) throws IOException {
    	FileInputStream in = new FileInputStream(name);
		byte buffer[] = new byte[in.available()];
  		
  		// read the image into the buffer
		in.read(buffer);
		return buffer;
    }
    
    /**
     *	Sends the data contained in the byte array through the DataOutputStream.
     *	An int representing the size of the byte array will be sent first.  The
     *	actual data contained in the array will follow.
     *
     *	@param	ops		the DataOutputStream 
     *			buf[]	the byte array to send
     *	@return			void
     *	@exception	IOException
     */
    
    public void sendFile(DataOutputStream ops, byte buf[]) throws IOException {
    	int length = buf.length;
   		
   		ops.flush();
    	ops.writeInt(length);
    	ops.write(buf, 0, length);
    }
}